<?php

/**
 * The admin-specific functionality of the plugin.
 *
 * @link       https://portail.hainaut.be/institution/direction-generale-des-systemes-dinformation
 * @since      1.0.0
 *
 * @package    Dgsi_Infolettre_Images_Sizes
 * @subpackage Dgsi_Infolettre_Images_Sizes/admin
 */

/**
 * The admin-specific functionality of the plugin.
 *
 * Defines the plugin name, version, and two examples hooks for how to
 * enqueue the admin-specific stylesheet and JavaScript.
 *
 * @package    Dgsi_Infolettre_Images_Sizes
 * @subpackage Dgsi_Infolettre_Images_Sizes/admin
 * @author     Julien Delneufcourt (Direction Générale des Systèmes d'Information) <julien.delneufcourt@hainaut.be>
 */
class Dgsi_Infolettre_Images_Sizes_Admin {

	/**
	 * The ID of this plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 * @var      string    $plugin_name    The ID of this plugin.
	 */
	private $plugin_name;

	/**
	 * The version of this plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 * @var      string    $version    The current version of this plugin.
	 */
	private $version;
  
  /**
   * The cache key to use for update process
   *
   * @since    1.0.0
   * @access   protected
   * @var      string    $cache_key    The key to retrieve the transient
   */
  protected $cache_key;
  
  /**
   * Allowes cache for plugin update
   *
   * @since    1.0.0
   * @access   protected
   * @var      bool    Allowes cache for plugin update
   */
  protected $cache_allowed;

	/**
	 * Initialize the class and set its properties.
	 *
	 * @since    1.0.0
	 * @param      string    $plugin_name       The name of this plugin.
	 * @param      string    $version    The version of this plugin.
	 */
	public function __construct( $plugin_name, $version ) {

		$this->plugin_name = $plugin_name;
		$this->version = $version;
    
    $this->cache_key = "{$plugin_name}-upd";
    $this->cache_allowed = false;

	}

	/**
	 * Register the stylesheets for the admin area.
	 *
	 * @since    1.0.0
	 */
	public function enqueue_styles() {

		/**
		 * This function is provided for demonstration purposes only.
		 *
		 * An instance of this class should be passed to the run() function
		 * defined in Dgsi_Infolettre_Images_Sizes_Loader as all of the hooks are defined
		 * in that particular class.
		 *
		 * The Dgsi_Infolettre_Images_Sizes_Loader will then create the relationship
		 * between the defined hooks and the functions defined in this
		 * class.
		 */

		wp_enqueue_style( $this->plugin_name, plugin_dir_url( __FILE__ ) . 'css/dgsi-infolettre-images-sizes-admin.css', array(), $this->version, 'all' );

	}

	/**
	 * Register the JavaScript for the admin area.
	 *
	 * @since    1.0.0
	 */
	public function enqueue_scripts() {

		/**
		 * This function is provided for demonstration purposes only.
		 *
		 * An instance of this class should be passed to the run() function
		 * defined in Dgsi_Infolettre_Images_Sizes_Loader as all of the hooks are defined
		 * in that particular class.
		 *
		 * The Dgsi_Infolettre_Images_Sizes_Loader will then create the relationship
		 * between the defined hooks and the functions defined in this
		 * class.
		 */

		wp_enqueue_script( $this->plugin_name, plugin_dir_url( __FILE__ ) . 'js/dgsi-infolettre-images-sizes-admin.js', array( 'jquery' ), $this->version, false );

	}
  
  /**
   * Ajoute les tailles pour les images 1, 2 et 3 colonnes et les crop au centre
   */
  public function add_new_image_sizes() {
    add_image_size( 'thumbnail-infolettre-1col', 550, 230, array( 'center', 'center' ) );
    add_image_size( 'thumbnail-infolettre-2cols', 270, 150, array( 'center', 'center' ) );
    add_image_size( 'thumbnail-infolettre-3cols', 180, 100, array( 'center', 'center' ) );
  }
  
  /**
   * Upscale thumbnails smaller than the defined image sizes
   *
   * @param $default
   * @param $orig_w
   * @param $orig_h
   * @param $new_w
   * @param $new_h
   * @param $crop
   * @return int[]|null
   */
  public function thumbnail_upscale( $default, $orig_w, $orig_h, $new_w, $new_h, $crop ){
    if ( !$crop ) return null; // let the WordPress default function handle this
    
    $aspect_ratio = $orig_w / $orig_h;
    $size_ratio = max($new_w / $orig_w, $new_h / $orig_h);
    
    $crop_w = round($new_w / $size_ratio);
    $crop_h = round($new_h / $size_ratio);
    
    $s_x = floor( ($orig_w - $crop_w) / 2 );
    $s_y = floor( ($orig_h - $crop_h) / 2 );
    
    return array( 0, 0, (int) $s_x, (int) $s_y, (int) $new_w, (int) $new_h, (int) $crop_w, (int) $crop_h );
  }
  
  public function settings_init() {
    // Register a new setting for "dgsi-infolettre-images-sizes" page.
    register_setting( 'dgsi-infolettre-images-sizes', 'diis_options' );
    
    // Register new sections in the "diis" page.
    add_settings_section(
      'diis_section_options_updates',
      __( 'Credential to be able to perform automatic updates', 'dgsi-infolettre-images-sizes' ),
      array( $this, 'section_options_callback' ),
      'dgsi-infolettre-images-sizes'
    );
    
    // Register new fields in the "diis_section_options_updates" section, inside the "diis" page.
    add_settings_field(
      'diis_field_key', // As of WP 4.6 this value is used only internally.
      // Use $args' label_for to populate the id inside the callback.
      __( 'Authorization key', 'dgsi-infolettre-images-sizes' ),
      array( $this, 'field_callback' ),
      'dgsi-infolettre-images-sizes',
      'diis_section_options_updates',
      array(
        'label_for'         => 'diis_field_key',
        'class'             => 'diis_row',
      )
    );
  }
  
  /**
   * Options section callback function.
   *
   * @param array $args  The settings array, defining title, id, callback.
   */
  public function section_options_callback( $args ) {
    ?>
    <p id="<?php echo esc_attr( $args['id'] ); ?>" style="display: none;"><?php esc_html_e( 'Configure the DGSI Infolettre Images Sizes',
        'dgsi-infolettre-images-sizes' );
      ?></p>
    <?php
  }
  
  /**
   * Requesting App field callback function.
   *
   * WordPress has magic interaction with the following keys: label_for, class.
   * - the "label_for" key value is used for the "for" attribute of the <label>.
   * - the "class" key value is used for the "class" attribute of the <tr> containing the field.
   * Note: you can add custom key value pairs to be used inside your callbacks.
   *
   * @param array $args
   */
  public function field_callback( $args ) {
    // Get the value of the settings registered with register_setting()
    $diis_options = get_option( 'diis_options' );
    
    switch ($args['label_for']) {
      
      case 'diis_field_key': ?>
        <input type="text"
               id="<?php echo esc_attr( $args['label_for'] ); ?>"
               class="regular-text" name="diis_options[<?php echo esc_attr( $args['label_for'] ); ?>]"
               value="<?php echo isset( $diis_options[ $args['label_for'] ] ) ? ( $diis_options[ $args['label_for'] ]
               ) : ( '' );?>"
        />
        <p class="description">
        <?php esc_html_e( 'The authorization key you received to be able to perform automatic updates.', 'dgsi-infolettre-images-sizes' ); ?>
        </p><?php
        break;
      
      default:
        break;
    }
  }
  
  /**
   * Add the top level menu page.
   */
  public function menu_page() {
    add_menu_page(
      'DGSI Infolettre Images Sizes Options',
      'DGSI Infolettre Images Sizes',
      'manage_options',
      'dgsi-infolettre-images-sizes',
      array( $this, 'options_page_html' ),
      'data:image/svg+xml;base64,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'
    );
  }
  
  /**
   * Top level menu callback function
   */
  public function options_page_html() {
    // check user capabilities
    if ( ! current_user_can( 'manage_options' ) ) {
      wp_die( __( 'You do not have sufficient permissions to access this page.' ) );
    }
    
    // add error/update messages
    
    // check if the user have submitted the settings
    // WordPress will add the "settings-updated" $_GET parameter to the url
    if ( isset( $_GET['settings-updated'] ) ) {
      // add settings saved message with the class of "updated"
      add_settings_error( 'diis_messages', 'diis_message', __( 'Settings Saved', 'dgsi-infolettre-images-sizes' ), 'updated' );
    }
    
    // show error/update messages
    settings_errors( 'diis_messages' );
    ?>
    <div class="wrap">
      <h1><?php echo esc_html( get_admin_page_title() ); ?></h1>
      <form action="options.php" method="post">
        <?php
        // output security fields for the registered setting "dgsi-infolettre-images-sizes"
        settings_fields( 'dgsi-infolettre-images-sizes' );
        // output setting sections and their fields
        // (sections are registered for "dgsi-infolettre-images-sizes", each field is registered to a specific section)
        do_settings_sections( 'dgsi-infolettre-images-sizes' );
        // output save settings button
        submit_button( __( 'Save Settings', 'dgsi-infolettre-images-sizes' ) );
        ?>
      </form>
    </div>
    <?php
  }
  
  private function request(){
    
    $remote = get_transient( $this->cache_key );
    
    if( false === $remote || ! $this->cache_allowed ) {
      
      if ( !empty( get_option( 'diis_options' )[ 'diis_field_key' ] ) ) {
        
        $remote = wp_remote_get(
          add_query_arg(
            [
              'p' => $this->plugin_name,
              'k' => get_option( 'diis_options' )[ 'diis_field_key' ],
              'r' => 'json'
            ],
            DIIS_UPDATE_BASE_URL . 'plugin-infos.php'
          ),
          [
            'timeout' => 10,
            'headers' => array('Accept' => 'application/json')
          ]
        );
        
        if ( is_wp_error($remote)
          || 200 !== wp_remote_retrieve_response_code( $remote )
          || empty( wp_remote_retrieve_body( $remote ) ) ) {
          return false;
        }
        
        set_transient( $this->cache_key, $remote, DAY_IN_SECONDS );
        
      } else {
        return false;
      }
      
    }
    
    $remote = json_decode( wp_remote_retrieve_body( $remote ) );
    
    return $remote;
    
  }
  
  public function info( $res, $action, $args ) {
    
    // do nothing if it is not our plugin
    if( $args->slug != $this->plugin_name ) {
      return $res;
    }
    
    // do nothing if you're not getting plugin information right now
    if( $action != 'plugin_information' ) {
      return $res;
    }
    
    // get updates
    $remote = $this->request();
    
    if( ! $remote ) {
      return false;
    }
    
    $remote->sections = (array) $remote->sections;
    $remote->icons = (array) $remote->icons;
    $remote->banners = (array) $remote->banners;
    
    return $remote;
    
  }
  
  private function update_exists() {
    
    $remote = $this->request();
    
    if ( $remote
      && version_compare( $this->version, $remote->version, '<' )
      && version_compare( $remote->requires, get_bloginfo( 'version' ), '<' )
      && version_compare( $remote->requires_php, PHP_VERSION, '<' )
    )
      return true;
    
    return false;
    
  }
  
  public function update( $transient ) {
    
    if ( empty( $transient->checked ) ) {
      return $transient;
    }
    
    $remote = $this->request();
    
    if( $remote && $this->update_exists() ) {
      $res = new stdClass();
      $res->slug = $this->plugin_name;
      $res->plugin = "{$this->plugin_name}/{$this->plugin_name}.php";
      $res->new_version = $remote->version;
      $res->tested = $remote->tested;
      $res->package = $remote->download_link;
      
      $transient->response[ $res->plugin ] = $res;
      
    }
    
    return $transient;
    
  }
  
  public function purge( $upgrader_object, $options ){
    
    if (
      $this->cache_allowed
      && 'update' === $options['action']
      && 'plugin' === $options[ 'type' ]
    ) {
      // just clean the cache when new plugin version is installed
      delete_transient( $this->cache_key );
    }
    
  }
  
  public function plugin_meta_links( $links, $file ) {
    $base_url = DIIS_UPDATE_BASE_URL;
    $key = get_option( 'diis_options' )['diis_field_key'];
    $langcode = get_locale();
    
    $details_link = "<a class=\"thickbox open-plugin-details-modal\"
        href=\"{$base_url}plugin-infos.php?p={$this->plugin_name}&k={$key}&l={$langcode}&r=html&TB_iframe=true&width=772&height=658\"
        data-title=\"DGSI Infolettre Images Sizes\"
        aria-label=\"" . __("More information about DGSI Infolettre Images Sizes", 'dgsi-infolettre-images-sizes') . "\">" .
      __("View details", 'dgsi-infolettre-images-sizes') . "</a>";
    
    if ( strpos( $file, $this->plugin_name ) !== false ) {
      if (!$this->update_exists()) {
        $links[] = $details_link;
      }
    }
    
    return $links;
  }

}
